// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

var utils = require('utils');

/**
 * Enum of possible key types (defined in WebCrypto.KeyType). Symmetric keys are
 * represented with the 'secret' type.
 * @enum {string}
 */
var KeyType =
    {__proto__: null, public: 'public', private: 'private', secret: 'secret'};

/**
 * Enum of possible key usages (subset of WebCrypto.KeyUsage).
 * @enum {string}
 */
var KeyUsage = {
  __proto__: null,
  sign: 'sign',
  verify: 'verify'
};

function CreateInvalidKeyObjectError() {
  return new Error('Invalid key object.');
}

/**
 * Implementation of WebCrypto.CryptoKey used in enterprise.platformKeys.
 * @param {KeyType} type The type of the new key.
 * @param {ArrayBuffer} keyIdentifier The key identifier. For asymmetric keys,
 * it corresponds to the Subject Public Key Info (SPKI) in DER encoding. For
 * symmetric keys, it corresponds to the unique internally generated `symKeyId`.
 * @param {KeyAlgorithm} algorithm The algorithm identifier.
 * @param {KeyUsage[]} usages The allowed key usages.
 * @param {boolean} extractable Whether the key is extractable.
 * @constructor
 */
function KeyImpl(type, keyIdentifier, algorithm, usages, extractable) {
  this.type = type;
  this.keyIdentifier = keyIdentifier;
  this.algorithm = algorithm;
  this.usages = usages;
  this.extractable = extractable;
}
$Object.setPrototypeOf(KeyImpl.prototype, null);

/**
 * The public base class of Key.
 */
function KeyBase() {}
KeyBase.prototype = {
  constructor: KeyBase,
  get algorithm() {
    return utils.deepCopy(privates(this).impl.algorithm);
  },
};

function Key() {
  privates(Key).constructPrivate(this, arguments);
}
utils.expose(Key, KeyImpl, {
  superclass: KeyBase,
  readonly: [
    'extractable',
    'type',
    'usages',
  ],
});

/**
 * Returns the identifier of a given `key`. If `key` is asymmetric, the returned
 * value represents the Subject Public Key Info. If `key` is symmetric, it
 * represents the key ID, generated by the internal API.
 * @param {Key} key
 * @return {ArrayBuffer} The key identifier (SPKI or symKeyId).
 */
function getKeyIdentifier(key) {
  if (!privates(key)) {
    throw CreateInvalidKeyObjectError();
  }
  var keyImpl = privates(key).impl;
  if (!keyImpl || !keyImpl.keyIdentifier) {
    throw CreateInvalidKeyObjectError();
  }
  return keyImpl.keyIdentifier;
}

exports.$set('Key', Key);
exports.$set('KeyType', KeyType);
exports.$set('KeyUsage', KeyUsage);
exports.$set('getKeyIdentifier', getKeyIdentifier);
